<?php
class Collection implements Iterator, Countable
{
	/**
	 * @var string Object class name
	 */
	protected $classname;

	/**
	 * @var int
	 */
	protected $lang;

	/**
	 * @var array Object definition
	 */
	protected $definition = array();

	/**
	 * @var DbQuery
	 */
	protected $query;

	/**
	 * @var array Collection of objects in an array
	 */
	protected $results = array();

	/**
	 * @var int Current object iteration
	 */
	protected $iterator = 0;

	/**
	 * @var bool Is current collection already hydrated
	 */
	protected $is_hydrated = false;

	/**
	 * @param string $classname
	 * @param int $lang
	 */
	public function __construct($classname, $lang = null)
	{
		$this->classname = $classname;
		$this->lang = $lang;

		$this->definition = ObjectModelDB::getDefinition($this->classname);
		if (!isset($this->definition['table']))
			throw new Exception('Miss table in definition for class '.$this->classname);
		else if (!isset($this->definition['primary']))
			throw new Exception('Miss primary in definition for class '.$this->classname);

		$this->query = new DbQuery();
		$this->query->select('a.*');
		$this->query->from($this->definition['table'], 'a');

		// If multilang, create association to lang table
		if (isset($this->definition['multilang']) && $this->definition['multilang'])
		{
			$this->query->select('b.*');
			$this->query->leftJoin($this->definition['table'].'_lang', 'b', 'a.'.$this->definition['primary'].' = b.'.$this->definition['primary']);
			$this->query->where("b.`lang` = '".$this->lang."'");
		}
	}
	
	/**
	 * Add LEFT JOIN restriction on query
	 *
	 * @param string $str
	 * @return Collection
	 */
	public function leftJoin($table, $alias, $join)
	{
		$this->query->leftJoin($table, $alias, $join);
		return $this;
	}

	/**
	 * Add WHERE restriction on query
	 *
	 * @param string $str
	 * @return Collection
	 */
	public function where($str)
	{
		$this->query->where($str);
		return $this;
	}

	/**
	 * Add ORDER BY restriction on query
	 *
	 * @param string $str
	 * @return Collection
	 */
	public function orderBy($str)
	{
		$this->query->orderBy($str);
		return $this;
	}

	/**
	 * Add GROUP BY restriction on query
	 *
	 * @param string $str
	 * @return Collection
	 */
	public function groupBy($str)
	{
		$this->query->groupBy($str);
		return $this;
	}
	
	/**
	 * Add LIMIT restriction on query
	 *
	 * @param string $str
	 * @return Collection
	 */
	public function limit($limit, $offset = 0)
	{
		$this->query->limit($limit, $offset);
		return $this;
	}

	/**
	 * Launch sql query to create collection of objects
	 *
	 * @param bool $display_query If true, query will be displayed (for debug purpose)
	 * @return Collection
	 */
	public function getAll($display_query = false)
	{
		if ($this->is_hydrated)
			return $this;
		$this->is_hydrated = true;

		if ($display_query)
			echo $this->query.'<br />';

		$this->results = Db::getInstance()->executeS($this->query);
		$this->results = ObjectModelDB::hydrateCollection($this->classname, $this->results, $this->lang);

		return $this;
	}

	/**
	 * This method is called when a foreach begin
	 */
	public function rewind()
	{
		$this->iterator = 0;
		$this->getAll();
	}

	/**
	 * Get current result
	 */
	public function current()
	{
		return $this->results[$this->iterator];
	}

	/**
	 * Check if there is a current result
	 */
	public function valid()
	{
		return isset($this->results[$this->iterator]);
	}

	/**
	 * Get current result index
	 */
	public function key()
	{
		return $this->iterator;
	}

	/**
	 * Go to next result
	 */
	public function next()
	{
		$this->iterator++;
	}

	/**
	 * Get total of results
	 *
	 * @return int
	 */
	public function count()
	{
		$this->getAll();
		return count($this->results);
	}
}