<?php
abstract class ObjectModelDB
{
	/** @var integer Object id */
	public $id;

	/** @var integer lang id */
	protected $lang = null;

	/**
	 * @var string SQL This property shouldn't be overloaded anymore in class, use static $definition['table'] property instead
	 * @deprecated
	 */
	protected $table;

	/**
	 * @var string SQL This property shouldn't be overloaded anymore in class, use static $definition['primary'] property instead
	 * @deprecated
	 */
	protected $identifier;

	/** @var array Required fields for admin panel forms */
 	protected $fieldsRequired = array();

	/** @var fieldsRequiredDatabase */
	protected static $fieldsRequiredDatabase = null;
	
	/** @var array Required fields Array for admin panel forms */
 	protected $fieldsArray = array();
	
	/** @var array Fields attributes (XML) */
 	protected $fieldsAttributes = array();

 	/** @var array Maximum fields size for admin panel forms */
 	protected $fieldsSize = array();

 	/** @var array Fields validity functions for admin panel forms */
 	protected $fieldsValidate = array();

	/** @var array Multilingual required fields for admin panel forms */
 	protected $fieldsRequiredLang = array();

 	/** @var array Multilingual maximum fields size for admin panel forms */
 	protected $fieldsSizeLang = array();

 	/** @var array Multilingual fields validity functions for admin panel forms */
 	protected $fieldsValidateLang = array();

	/** @var array tables */
 	protected $tables = array();

	protected static $_cache = array();

	/** @var  string path to image directory. Used for image deletion. */
	protected $image_dir = null;

	/** @var string file type of image files. Used for image deletion. */
	protected $image_format = 'jpg';

	/**
	 * @var array Contain object definition
	 * @since 1.5.0
	 */
	public static $definition = array();

	/**
	 * @var array Contain current object definition
	 */
	protected $def;

	/**
	 * Build object
	 *
	 * @param integer $id Existing object id in order to load object (optional)
	 * @param integer $lang Required if object is multilingual (optional)
	 */
	public function __construct($id = null, $lang = null)
	{
		$this->def = self::getDefinition($this);
		if (isset($this->def['table']))
			$this->table = $this->def['table'];
		if (isset($this->def['primary']))
			$this->identifier = $this->def['primary'];
			
		/*if (!is_null($lang))
			$this->lang = (Language::getLanguage($lang) !== false) ? $lang : 2;*/

		if ($id)
		{
			// Load object from database if object id is present
			if (!isset(self::$_cache[$this->table][(int)$id][$lang]))
			{
				$sql = 'SELECT *
						FROM `'._DB_PREFIX_.$this->table.'` a '.
						($lang ? ('LEFT JOIN `'.pSQL(_DB_PREFIX_.$this->table).'_lang` b ON (a.`'.$this->identifier.'` = b.`'.$this->identifier).'` AND `lang` = "'.($lang).'")' : '')
						.' WHERE 1 AND a.`'.$this->identifier.'` = '.(int)$id;
				self::$_cache[$this->table][(int)($id)][$lang] = Db::getInstance()->getRow($sql);
			}

			$result = self::$_cache[$this->table][(int)$id][$lang];
			if ($result)
			{
				$this->id = (int)($id);
				foreach ($result AS $key => $value)
					if (key_exists($key, $this))
						$this->{$key} = $value;

				if (!$lang AND method_exists($this, 'getTranslationsFieldsChild'))
				{
					$sql = 'SELECT * FROM `'.pSQL(_DB_PREFIX_.$this->table).'_lang`
							WHERE `'.$this->identifier.'` = '.(int)$id;
					$result = Db::getInstance()->executeS($sql);
					if ($result)
						foreach ($result as $row)
							foreach ($row AS $key => $value)
							{
								if (key_exists($key, $this) AND $key != $this->identifier)
								{
									if (!is_array($this->{$key}))
										$this->{$key} = array();

									// @Todo: stripslashes() MUST BE removed in 1.4.6 and later, but is kept in 1.4.5 for a compatibility issue
									$this->{$key}[$row['lang']] = stripslashes($value);
								}
							}
				}
			}
		}
	}
	
		/**
	 * Returns object validation rules (fields validity)
	 *
	 * @param string $className Child class name for static use (optional)
	 * @return array Validation rules (fields validity)
	 */
	public static function getValidationRules($className = __CLASS__)
	{
		$object = new $className();
		return array(
		'attributes' => $object->fieldsAttributes,
		'array' => $object->fieldsArray,
		'required' => $object->fieldsRequired,
		'size' => $object->fieldsSize,
		'validate' => $object->fieldsValidate,
		'requiredLang' => $object->fieldsRequiredLang,
		'sizeLang' => $object->fieldsSizeLang,
		'validateLang' => $object->fieldsValidateLang);
	}
	
	/**
	 * Prepare fields for ObjectModel class (add, update)
	 * All fields are verified (pSQL, intval...)
	 *
	 * @return array All object fields
	 */
	public function getFields()
	{
		$this->validateFields();
		$fields = array();
		
		// Set primary key in fields
		if (isset($this->id))
			$fields[$this->def['primary']] = $this->id;
		
		foreach ($this->fieldsValidate as $field => $value)
		{
			if (!Validate::isTableOrIdentifier($field))
					die(Tools::displayError());
			$fields[$field] = pSQL($this->$field);
		}
		
		// Ensure that we get something to insert
		if (!$fields)
			$fields[$this->def['primary']] = $this->id;
		
		return $fields;
	}
	
	/**
	* Check then return multilingual fields for database interaction
	*
	* @return array Multilingual fields
	*/
	public function getTranslationsFieldsChild()
	{
		$this->validateFieldsLang();

		global $listLang;
		
		$fields = array();
		foreach ($listLang as $language)
		{
			foreach ($this->fieldsValidateLang as $field => $value)
			{
				if (!Validate::isTableOrIdentifier($field))
					die(Tools::displayError());

				/* Check fields validity */
				if (isset($this->{$field}[$language]))
					$fields[$language][$field] = pSQL($this->{$field}[$language]);
			}
		}
		return $fields;
	}

	/**
	 * Save current object to database (add or update)
	 *
	 * return boolean Insertion result
	 */
	public function save($nullValues = false, $autodate = true)
	{
		return (int)($this->id) > 0 ? $this->update($nullValues) : $this->add($autodate, $nullValues);
	}

	/**
	 * Add current object to database
	 *
	 * return boolean Insertion result
	 */
	public function add($autodate = true, $nullValues = false)
	{
	 	if (!Validate::isTableOrIdentifier($this->table))
			throw new Exception('not table or identifier : '.$this->table);

		/* Automatically fill dates */
		if ($autodate AND key_exists('date_add', $this))
			$this->date_add = date('Y-m-d H:i:s');
		if ($autodate AND key_exists('date_upd', $this))
			$this->date_upd = date('Y-m-d H:i:s');
		/* Database insertion */
		if ($nullValues)
			$result = Db::getInstance()->autoExecuteWithNullValues(_DB_PREFIX_.$this->table, $this->getFields(), 'INSERT');
		else
			$result = Db::getInstance()->insert(_DB_PREFIX_.$this->table, $this->getFields(), 'INSERT');

		if (!$result)
			return false;

		/* Get object id in database */
		$this->id = Db::getInstance()->Insert_ID();
		/* Database insertion for multilingual fields related to the object */
		if (method_exists($this, 'getTranslationsFieldsChild'))
		{
			$fields = $this->getTranslationsFieldsChild();
			if ($fields AND is_array($fields))
				foreach ($fields AS $lang => &$field)
				{
					foreach (array_keys($field) AS $key)
					 	if (!Validate::isTableOrIdentifier($key))
			 				throw new Exception('key '.$key.' is not table or identifier, ');
							
					$field[$this->identifier] = (int)$this->id;
					$field['lang'] = $lang;
					
					$result &= Db::getInstance()->AutoExecute(_DB_PREFIX_.$this->table.'_lang', $field, 'INSERT');
				}
		}

		return $result;
	}

	/**
	 * Update current object to database
	 *
	 * @return boolean Update result
	 */
	public function update($nullValues = false)
	{
	 	if (!Validate::isTableOrIdentifier($this->identifier) OR !Validate::isTableOrIdentifier($this->table))
			throw new Exception('wrong identifier or table:'.$this->identifier.', table: '.$this->table);

		$this->clearCache();
		/* Automatically fill dates */
		if (key_exists('date_upd', $this))
			$this->date_upd = date('Y-m-d H:i:s');
		
		/* Database update */
		if ($nullValues)
			$result = Db::getInstance()->autoExecuteWithNullValues(_DB_PREFIX_.$this->table, $this->getFields(), 'UPDATE', '`'.pSQL($this->identifier).'` = '.(int)($this->id));
		else
			$result = Db::getInstance()->autoExecute(_DB_PREFIX_.$this->table, $this->getFields(), 'UPDATE', '`'.pSQL($this->identifier).'` = '.(int)($this->id));
		if (!$result)
			return false;

		// Database update for multilingual fields related to the object
		if (method_exists($this, 'getTranslationsFieldsChild'))
		{
			$fields = $this->getTranslationsFieldsChild();
			if (is_array($fields))
			{
				foreach ($fields as $lang => $field)
				{
					foreach (array_keys($field) as $key)
						if (!Validate::isTableOrIdentifier($key))
							throw new Exception('key '.$key.' is not a valid table or identifier');

					$where = pSQL($this->identifier).' = '.(int)$this->id
								.' AND lang = "'.$lang.'"';
					if (Db::getInstance()->getValue('SELECT COUNT(*) FROM '.pSQL(_DB_PREFIX_.$this->table).'_lang WHERE '.$where))
						$result &= Db::getInstance()->AutoExecute(_DB_PREFIX_.$this->table.'_lang', $field, 'UPDATE', $where);
					else
					{
						$field[$this->identifier] = (int)$this->id;
						$field['lang'] = $lang;
						
						$result &= Db::getInstance()->AutoExecute(_DB_PREFIX_.$this->table.'_lang', $field, 'INSERT');
					}
				}
			}
		}

		return $result;
	}

	/**
	 * Delete current object from database
	 *
	 * return boolean Deletion result
	 */
	public function delete()
	{
	 	if (!Validate::isTableOrIdentifier($this->identifier) OR !Validate::isTableOrIdentifier($this->table))
			throw new Exception('wrong identifier or table:'.$this->identifier.', table: '.$this->table);

		$this->clearCache();

		/* Database deletion */
		$result = Db::getInstance()->execute('DELETE FROM `'.pSQL(_DB_PREFIX_.$this->table).'` WHERE `'.pSQL($this->identifier).'` = '.(int)($this->id));
		if (!$result)
			return false;

		/* Database deletion for multilingual fields related to the object */
		if (method_exists($this, 'getTranslationsFieldsChild'))
			Db::getInstance()->execute('DELETE FROM `'.pSQL(_DB_PREFIX_.$this->table).'_lang` WHERE `'.pSQL($this->identifier).'` = '.(int)($this->id));

		return $result;
	}

	/**
	 * Delete several objects from database
	 *
	 * return boolean Deletion result
	 */
	public function deleteSelection($selection)
	{
		if (!is_array($selection) OR !Validate::isTableOrIdentifier($this->identifier) OR !Validate::isTableOrIdentifier($this->table))
			throw new Exception('selection is not an array, or identifier or table:'.$this->identifier.', table: '.$this->table);
		$result = true;
		foreach ($selection AS $id)
		{
			$this->id = (int)($id);
			$result = $result AND $this->delete();
		}
		return $result;
	}

	/**
	 * Toggle object status in database
	 *
	 * return boolean Update result
	 */
	public function toggleStatus()
	{
	 	if (!Validate::isTableOrIdentifier($this->identifier) OR !Validate::isTableOrIdentifier($this->table))
			throw new Exception('identifier or table:'.$this->identifier.', table: '.$this->table);

	 	/* Object must have a variable called 'active' */
	 	elseif (!key_exists('active', $this))
			throw new Exception('property "active is missing in object '.get_class($this));

	 	/* Update active status on object */
	 	$this->active = (int)(!$this->active);

		/* Change status to active/inactive */
		return Db::getInstance()->execute('
		UPDATE `'.pSQL(_DB_PREFIX_.$this->table).'`
		SET `active` = !`active`
		WHERE `'.pSQL($this->identifier).'` = '.(int)($this->id));
	}

	/**
	 * Prepare multilingual fields for database insertion
	 *
	 * @param array $fieldsArray Multilingual fields to prepare
	 * return array Prepared fields for database insertion
	 */
	protected function getTranslationsFields($fieldsArray)
	{
		global $listLang;
		
		/* WARNING : Product do not use this function, so do not forget to report any modification if necessary */
	 	if (!Validate::isTableOrIdentifier($this->identifier))
	 		throw new Exception('identifier is not table or identifier : '.$this->identifier);

		$fields = array();

		if($this->lang == NULL)
			foreach ($listLang as $language)
				$this->makeTranslationFields($fields, $fieldsArray, $language);
		else
			$this->makeTranslationFields($fields, $fieldsArray, $this->lang);

		return $fields;
	}

	protected function makeTranslationFields(&$fields, &$fieldsArray, $language)
	{
		$fields[$language]['lang'] = $language;
		$fields[$language][$this->identifier] = (int)($this->id);
		foreach ($fieldsArray as $k => $field)
		{
			$html = false;
			$fieldName = $field;
			if (is_array($field))
			{
				$fieldName = $k;
				$html = (isset($field['html'])) ? $field['html'] : false;
			}

			/* Check fields validity */
			if (!Validate::isTableOrIdentifier($fieldName))
				throw new Exception('identifier is not table or identifier : '.$fieldName);

			/* Copy the field, or the default language field if it's both required and empty */
			if ((!$this->lang AND isset($this->{$fieldName}[$language]) AND !empty($this->{$fieldName}[$language]))
			OR ($this->lang AND isset($this->$fieldName) AND !empty($this->$fieldName)))
				$fields[$language][$fieldName] = $this->lang ? pSQL($this->$fieldName, $html) : pSQL($this->{$fieldName}[$language], $html);
			elseif (in_array($fieldName, $this->fieldsRequiredLang))
				$fields[$language][$fieldName] = $this->lang ? pSQL($this->$fieldName, $html) : pSQL($this->{$fieldName}[Configuration::get('PS_LANG_DEFAULT')], $html);
			else
				$fields[$language][$fieldName] = '';
		}
	}

	/**
	 * Check for fields validity before database interaction
	 */
	public function validateFields($die = true, $errorReturn = false)
	{
		foreach ($this->fieldsRequired as $field)
			if (Tools::isEmpty($this->{$field}) AND (!is_numeric($this->{$field})))
			{
				if ($die) die (Tools::displayError().' ('.get_class($this).' -> '.$field.' is empty)');
				return $errorReturn ? get_class($this).' -> '.$field.' is empty' : false;
			}
		foreach ($this->fieldsSize as $field => $size)
			if (isset($this->{$field}) AND Tools::strlen($this->{$field}) > $size)
			{
				if ($die) die (Tools::displayError().' ('.get_class($this).' -> '.$field.' length > '.$size.')');
				return $errorReturn ? get_class($this).' -> '.$field.' length > '.$size : false;
			}
		$validate = new Validate();
		foreach ($this->fieldsValidate as $field => $method)
			if (!method_exists($validate, $method))
				die (Tools::displayError('validation function not found').' '.$method);
			elseif (!empty($this->{$field}) AND !call_user_func(array('Validate', $method), $this->{$field}))
			{
				if ($die) die (Tools::displayError().' ('.get_class($this).' -> '.$field.' = '.$this->{$field}.')');
				return $errorReturn ? get_class($this).' -> '.$field.' = '.$this->{$field} : false;
			}
		foreach ($this->fieldsAttributes as $field => $method)
			if (!method_exists($validate, $method))
				die (Tools::displayError('validation function not found').' '.$method);
			elseif (!empty($this->{$field}) AND !call_user_func(array('Validate', $method), $this->{$field}))
			{
				if ($die) die (Tools::displayError().' ('.get_class($this).' -> '.$field.' = '.$this->{$field}.')');
				return $errorReturn ? get_class($this).' -> '.$field.' = '.$this->{$field} : false;
			}
		return true;
	}

	/**
	 * Check for multilingual fields validity before database interaction
	 */
	public function validateFieldsLang($die = true, $errorReturn = false)
	{
		$defaultLanguage = _DEFAULT_LANG_;
		foreach ($this->fieldsRequiredLang as $fieldArray)
		{
			if (!is_array($this->{$fieldArray}))
				continue ;
			if (!$this->{$fieldArray} OR !sizeof($this->{$fieldArray}) OR ($this->{$fieldArray}[$defaultLanguage] !== '0' AND empty($this->{$fieldArray}[$defaultLanguage])))
			{
				if ($die) die (Tools::displayError().' ('.get_class($this).'->'.$fieldArray.' '.Tools::displayError('is empty for default language').')');
				return $errorReturn ? get_class($this).'->'.$fieldArray.' '.Tools::displayError('is empty for default language') : false;
			}
		}
		foreach ($this->fieldsSizeLang as $fieldArray => $size)
		{
			if (!is_array($this->{$fieldArray}))
				continue ;
			foreach ($this->{$fieldArray} as $k => $value)
				if (Tools::strlen($value) > $size)
				{
					if ($die) die (Tools::displayError().' ('.get_class($this).'->'.$fieldArray.' '.Tools::displayError('length >').' '.$size.' '.Tools::displayError('for language').')');
					return $errorReturn ? get_class($this).'->'.$fieldArray.' '.Tools::displayError('length >').' '.$size.' '.Tools::displayError('for language') : false;
				}
		}
		$validate = new Validate();
		foreach ($this->fieldsValidateLang as $fieldArray => $method)
		{
			if (!is_array($this->{$fieldArray}))
				continue ;
			foreach ($this->{$fieldArray} as $k => $value)
				if (!method_exists($validate, $method))
					die (Tools::displayError('validation function not found').' '.$method);
				elseif (!empty($value) AND !call_user_func(array('Validate', $method), $value))
				{
					if ($die) die (Tools::displayError('The following field is invalid according to the validate method ').'<b>'.$method.'</b>:<br/> ('.get_class($this).'->'.$fieldArray.' = '.$value.' '.Tools::displayError('for language').' '.$k.')');
					return $errorReturn ? Tools::displayError('The following field is invalid according to the validate method ').'<b>'.$method.'</b>:<br/> ('. get_class($this).'->'.$fieldArray.' = '.$value.' '.Tools::displayError('for language').' '.$k : false;
				}
		}
		return true;
	}
	
	static public function displayFieldName($field, $className = __CLASS__, $htmlentities = true)
	{
		global $_FIELDS;
		@include(_KAM_TRANSLATIONS_DIR_.'/fields.php');

		$key = $className.'_'.md5($field);
		return ((is_array($_FIELDS) AND array_key_exists($key, $_FIELDS)) ? ($htmlentities ? htmlentities($_FIELDS[$key], ENT_QUOTES, 'utf-8') : $_FIELDS[$key]) : $field);
	}

	public function clearCache($all = false)
	{
		if ($all AND isset(self::$_cache[$this->table]))
			unset(self::$_cache[$this->table]);
		elseif ($this->id AND isset(self::$_cache[$this->table][(int)$this->id]))
			unset(self::$_cache[$this->table][(int)$this->id]);
	}

	/**
	* Specify if an ObjectModel is already in database
	*
	* @param $id_entity entity id
	* @return boolean
	*/
	public static function existsInDatabase($id_entity, $table)
	{
		$row = Db::getInstance()->getRow('
		SELECT `id_'.$table.'` as id
		FROM `'._DB_PREFIX_.$table.'` e
		WHERE e.`id_'.$table.'` = '.(int)($id_entity));

		return isset($row['id']);
	}

	/**
	 * This method is allow to know if a entity is currently used
	 * @since 1.5.0.1
	 * @param string $table name of table linked to entity
	 * @param bool $has_active_column true if the table has an active column
	 * @return bool
	 */
	public static function isCurrentlyUsed($table, $has_active_column = false)
	{
		$query = new DbQuery();
		$query->select('`id_'.pSQL($table).'`');
		$query->from(pSQL($table));
		if ($has_active_column)
			$query->where('`active` = 1');
		return (bool)Db::getInstance()->getValue($query);
	}

	/**
	 * Get object identifier name
	 *
	 * @since 1.5.0
	 * @return string
	 */
	public function getIdentifier()
	{
		return $this->identifier;
	}

	/**
	 * Get list of fields related to language to validate
	 *
	 * @since 1.5.0
	 * @return array
	 */
	public function getFieldsValidateLang()
	{
		return $this->fieldsValidateLang;
	}

	/**
	 * Fill an object with given data. Data must be an array with this syntax: array(objProperty => value, objProperty2 => value, etc.)
	 *
	 * @since 1.5.0
	 * @param array $data
	 * @param int $lang
	 */
	public function hydrate(array $data, $lang = null)
	{
		$this->lang = $lang;
		if (isset($data[$this->identifier]))
			$this->id = $data[$this->identifier];
		foreach ($data as $key => $value)
			if (array_key_exists($key, $this))
				$this->$key = $value;
	}

	/**
	 * Fill (hydrate) a list of objects in order to get a collection of these objects
	 *
	 * @since 1.5.0
	 * @param string $class Class of objects to hydrate
	 * @param array $datas List of data (multi-dimensional array)
	 * @param int $lang
	 * @return array
	 */
	public static function hydrateCollection($class, array $datas, $lang = null)
	{
		if (!class_exists($class))
			throw new Exception("Class '$class' not found");

		$collection = array();
		$group_keys = array();
		$rows = array();
		$identifier = $fieldsValidateLang = null;
		if ($datas)
		{
			// Get identifier and lang fields
			$obj = new $class;
			if (!$obj instanceof ObjectModelDB)
				throw new Exception("Class '$class' must be an instance of class ObjectModel");
			$identifier = $obj->getIdentifier();
			$fieldsValidateLang = $obj->getFieldsValidateLang();

			// Check primary key
			if (!array_key_exists($identifier, $datas[0]))
				throw new Exception("Identifier '$identifier' not found for class '$class'");

			foreach ($datas as $row)
			{
				// Get object common properties
				$id = $row[$identifier];
				if (!isset($rows[$id]))
					$rows[$id] = $row;

				// Get object lang properties
				if (isset($row['lang']) && !$lang)
				{
					foreach ($fieldsValidateLang as $field => $validator)
					{
						if (!$lang)
						{
							if (!is_array($rows[$id][$field]))
								$rows[$id][$field] = array();
							$rows[$id][$field][$row['lang']] = $row[$field];
						}
					}
				}
			}
		}

		// Hydrate objects
		foreach ($rows as $row)
		{
			$obj = new $class;
			$obj->hydrate($row, $lang);
			$collection[] = $obj;
		}
		return $collection;
	}
	
	public static function getDefinition($class, $field = null)
	{
		$reflection = new ReflectionClass($class);
		$definition = $reflection->getStaticPropertyValue('definition');
		$definition['classname'] = $class;
		if ($field)
			return isset($definition[$field]) ? $definition[$field] : null;
		return $definition;
	}
}
